#!/usr/bin/env python3
import time
import tkinter as tk
import tkinter.font as tkfont
from datetime import datetime, timedelta


def pick_dseg7_family(root: tk.Tk) -> str:
    available = set(tkfont.families(root))
    for name in ("DSEG7 Classic Bold", "DSEG7 Classic", "DSEG7 Modern"):
        if name in available:
            return name
    return "DejaVu Sans Mono"


class TimeCircuitsClock(tk.Tk):
    def __init__(self) -> None:
        super().__init__()

        # --- behaviour ---
        CURSOR_HIDE_SECONDS = 7

        # --- tuning knobs ---
        AMPM_STACK_SEP_MULT = 6.15    # bigger = more vertical spacing between AM and PM lamps
        HDR_PAD_X = 16                # header red background padding (px)
        HDR_PAD_Y = 6
        AMPM_PAD_X = 7                # AM/PM red background padding (px)
        AMPM_PAD_Y = 3

        # black word-sized background behind DESTINATION/PRESENT/LAST labels
        TITLE_PAD_X = 18
        TITLE_PAD_Y = 6
        TITLE_BG_COLOR = "#101010"

        # spacing around ':' (you set this to 40)
        TIME_SIDE_GAP = 40

        # AM/PM horizontal centering tweak (scales with screen)
        AMPM_SHIFT_FRAC = 0.10  # 0.10 is what you found “spot on”

        self.AMPM_STACK_SEP_MULT = AMPM_STACK_SEP_MULT
        self.HDR_PAD_X = HDR_PAD_X
        self.HDR_PAD_Y = HDR_PAD_Y
        self.AMPM_PAD_X = AMPM_PAD_X
        self.AMPM_PAD_Y = AMPM_PAD_Y
        self.TITLE_PAD_X = TITLE_PAD_X
        self.TITLE_PAD_Y = TITLE_PAD_Y
        self.TITLE_BG_COLOR = TITLE_BG_COLOR
        self.TIME_SIDE_GAP = TIME_SIDE_GAP
        self.AMPM_SHIFT_FRAC = AMPM_SHIFT_FRAC

        self.configure(bg="#111111")
        self.overrideredirect(False)
        self.attributes("-fullscreen", True)
        self.attributes("-topmost", True)

        self.c = tk.Canvas(self, bg="#111111", highlightthickness=0, bd=0)
        self.c.pack(fill="both", expand=True)

        # track real size changes only (prevents jitter from spurious Configure events)
        self._last_w = None
        self._last_h = None

        # fonts
        dseg7 = pick_dseg7_family(self)
        self.f_hdr = tkfont.Font(family="DejaVu Sans", size=16, weight="bold")
        self.f_seg = tkfont.Font(family=dseg7, size=64, weight="bold")
        self.f_lbl = tkfont.Font(family="DejaVu Sans", size=14, weight="bold")
        self.f_title = tkfont.Font(family="DejaVu Sans", size=16, weight="bold")

        # colours
        self.COL_ORANGE = "#FF6A00"
        self.COL_GREEN = "#36FF3B"
        self.COL_YELLOW = "#FFCC33"
        self.COL_DIMTXT = "#CFCFCF"
        self.COL_PANEL = "#2A2A2A"
        self.COL_HDR = "#7B1E1E"
        self.COL_HDRTXT = "#F2F2F2"
        self.COL_BOLT = "#1A1A1A"
        self.COL_DARK = "#1F1F1F"

        # cursor hide/show
        self._cursor_hide_ms = int(CURSOR_HIDE_SECONDS * 1000)
        self._cursor_after_id = None
        self._cursor_hidden = False
        self.bind_all("<Motion>", self._on_mouse_motion)

        # quit controls
        self.after(50, self._arm_quit_controls)
        self._schedule_cursor_hide()

        # fixed destination
        self.destination_dt = datetime(1985, 10, 26, 1, 22)

        # UI objects
        self._items = {}
        self._build_ui()

        # initial layout AFTER fullscreen settles
        self.after(150, self._layout)

        # two independent loops:
        self.after(200, self._update_time)     # text + lamps
        self.after(1, self._blink_colons)      # only colons, synced

        # relayout on resize only (real size change)
        self.bind("<Configure>", self._on_configure)

    # ----------------- input -----------------

    def _arm_quit_controls(self) -> None:
        def _quit(_e=None):
            self.destroy()

        self.bind("<Escape>", _quit)
        self.bind("<KeyPress-Escape>", _quit)
        self.bind_all("<Escape>", _quit)
        self.bind_all("<KeyPress-Escape>", _quit)

        self.bind("<Button-3>", _quit)
        self.c.bind("<Button-3>", _quit)

    def _set_cursor_hidden(self, hidden: bool) -> None:
        if hidden and not self._cursor_hidden:
            self.configure(cursor="none")
            self.c.configure(cursor="none")
            self._cursor_hidden = True
        elif not hidden and self._cursor_hidden:
            self.configure(cursor="")
            self.c.configure(cursor="")
            self._cursor_hidden = False

    def _schedule_cursor_hide(self) -> None:
        if self._cursor_after_id is not None:
            try:
                self.after_cancel(self._cursor_after_id)
            except Exception:
                pass
        self._cursor_after_id = self.after(self._cursor_hide_ms, lambda: self._set_cursor_hidden(True))

    def _on_mouse_motion(self, _evt=None) -> None:
        self._set_cursor_hidden(False)
        self._schedule_cursor_hide()

    def _on_configure(self, _e=None) -> None:
        w = self.winfo_width()
        h = self.winfo_height()

        if w < 200 or h < 200:
            return

        if w == self._last_w and h == self._last_h:
            return

        self._last_w, self._last_h = w, h
        self.after_idle(self._layout)

    # ----------------- UI build -----------------

    def _build_ui(self) -> None:
        self._items["rows"] = []
        for title, color in (
            ("DESTINATION TIME", self.COL_ORANGE),
            ("PRESENT TIME", self.COL_GREEN),
            ("LAST TIME DEPARTED", self.COL_YELLOW),
        ):
            self._items["rows"].append(self._create_row(title, color))

    def _create_row(self, title: str, seg_color: str) -> dict:
        panel = self.c.create_rectangle(0, 0, 10, 10, fill=self.COL_PANEL, outline=self.COL_BOLT, width=2)

        # bottom strip behind the label
        title_bar = self.c.create_rectangle(0, 0, 10, 10, fill=self.COL_PANEL, outline=self.COL_PANEL)

        # black word-sized background behind the label text
        title_bg = self.c.create_rectangle(0, 0, 10, 10, fill=self.TITLE_BG_COLOR, outline=self.TITLE_BG_COLOR, width=0)
        title_txt = self.c.create_text(0, 0, text=title, fill=self.COL_DIMTXT, font=self.f_title, anchor="center")
        self.c.tag_raise(title_bg, title_bar)
        self.c.tag_lower(title_bg, title_txt)

        # headers: text + red background rect sized to word
        hdr_names = ["MONTH", "DAY", "YEAR", "HOUR", "MIN"]
        hdr = []
        for n in hdr_names:
            rect = self.c.create_rectangle(0, 0, 10, 10, fill=self.COL_HDR, outline=self.COL_HDR, width=0)
            text = self.c.create_text(0, 0, text=n, fill=self.COL_HDRTXT, font=self.f_hdr, anchor="center")
            self.c.tag_lower(rect, text)
            hdr.append((rect, text))

        v_month = self.c.create_text(0, 0, text="OCT", fill=seg_color, font=self.f_seg, anchor="center")
        v_day = self.c.create_text(0, 0, text="26", fill=seg_color, font=self.f_seg, anchor="center")
        v_year = self.c.create_text(0, 0, text="1985", fill=seg_color, font=self.f_seg, anchor="center")
        v_hour = self.c.create_text(0, 0, text="01", fill=seg_color, font=self.f_seg, anchor="center")
        v_min = self.c.create_text(0, 0, text="22", fill=seg_color, font=self.f_seg, anchor="center")

        # colon as two dots (blink)
        dot1 = self.c.create_oval(0, 0, 10, 10, fill=seg_color, outline=seg_color, width=0)
        dot2 = self.c.create_oval(0, 0, 10, 10, fill=seg_color, outline=seg_color, width=0)

        # AM/PM lamps + labels + red backgrounds (sized to word)
        am_bg = self.c.create_rectangle(0, 0, 10, 10, fill=self.COL_HDR, outline=self.COL_HDR, width=0)
        pm_bg = self.c.create_rectangle(0, 0, 10, 10, fill=self.COL_HDR, outline=self.COL_HDR, width=0)
        txt_am = self.c.create_text(0, 0, text="AM", fill=self.COL_HDRTXT, font=self.f_hdr, anchor="center")
        txt_pm = self.c.create_text(0, 0, text="PM", fill=self.COL_HDRTXT, font=self.f_hdr, anchor="center")
        self.c.tag_lower(am_bg, txt_am)
        self.c.tag_lower(pm_bg, txt_pm)

        lamp_am = self.c.create_oval(0, 0, 10, 10, fill=self.COL_DARK, outline="#000000")
        lamp_pm = self.c.create_oval(0, 0, 10, 10, fill=self.COL_DARK, outline="#000000")

        return {
            "panel": panel,
            "title_bar": title_bar,
            "title_bg": title_bg,
            "title_txt": title_txt,
            "hdr": hdr,  # [(rect,text), ...] in MONTH/DAY/YEAR/HOUR/MIN order
            "seg_color": seg_color,
            "month": v_month,
            "day": v_day,
            "year": v_year,
            "hour": v_hour,
            "min": v_min,
            "dot1": dot1,
            "dot2": dot2,
            "lamp_am": lamp_am,
            "lamp_pm": lamp_pm,
            "txt_am": txt_am,
            "txt_pm": txt_pm,
            "am_bg": am_bg,
            "pm_bg": pm_bg,
        }

    # ----------------- measure helpers -----------------

    def _bbox_wh(self, item_id: int) -> tuple[int, int]:
        bb = self.c.bbox(item_id)
        if not bb:
            return (0, 0)
        x0, y0, x1, y1 = bb
        return (max(0, x1 - x0), max(0, y1 - y0))

    def _measure_text(self, text: str, font: tkfont.Font) -> int:
        tid = self.c.create_text(-10000, -10000, text=text, font=font, anchor="nw")
        self.c.update_idletasks()
        w, _ = self._bbox_wh(tid)
        self.c.delete(tid)
        return w

    def _fit_bg_to_text(self, rect_id: int, text_id: int, pad_x: int, pad_y: int) -> None:
        bb = self.c.bbox(text_id)
        if not bb:
            return
        x0, y0, x1, y1 = bb
        self.c.coords(rect_id, x0 - pad_x, y0 - pad_y, x1 + pad_x, y1 + pad_y)

    # ----------------- layout -----------------

    def _layout(self) -> None:
        W = self.winfo_width()
        H = self.winfo_height()
        if W < 200 or H < 200:
            W = self.winfo_screenwidth()
            H = self.winfo_screenheight()

        pad = int(min(W, H) * 0.02)
        gap_y = int(H * 0.02)
        row_h = H / 3.0

        hdr_size = max(10, int(row_h * 0.060))
        lbl_size = max(9, int(row_h * 0.065))
        title_size = max(10, int(row_h * 0.0711))
        self.f_hdr.configure(size=hdr_size)
        self.f_lbl.configure(size=lbl_size)
        self.f_title.configure(size=title_size)

        col_fracs = [0.22, 0.12, 0.28, 0.20, 0.18]

        x_left = pad + int(W * 0.03)
        x_right = W - pad - int(W * 0.03)
        usable_w = max(1, x_right - x_left)

        col_widths = [int(usable_w * f) for f in col_fracs]
        col_widths[-1] = usable_w - sum(col_widths[:-1])

        bounds = []
        x = x_left
        for cw in col_widths:
            bounds.append((x, x + cw, x + cw // 2))  # (x0,x1,cx)
            x += cw

        # Fit big 7-seg font on resize only
        inner = max(12, int(W * 0.010))
        target_seg = max(28, int(row_h * 0.44))
        min_seg = 18

        seg_size = target_seg
        while True:
            self.f_seg.configure(size=seg_size)

            w_month = self._measure_text("SEP", self.f_seg)
            w_day = self._measure_text("88", self.f_seg)
            w_year = self._measure_text("2000", self.f_seg)
            w_hour = self._measure_text("88", self.f_seg)
            w_min = self._measure_text("88", self.f_seg)

            ok = True
            if w_month > (col_widths[0] - inner): ok = False
            if w_day > (col_widths[1] - inner): ok = False
            if w_year > (col_widths[2] - inner): ok = False
            if w_hour > (col_widths[3] - inner): ok = False
            if w_min > (col_widths[4] - inner): ok = False

            if ok or seg_size <= min_seg:
                break
            seg_size -= 2

        for i, row in enumerate(self._items["rows"]):
            y0 = int(i * row_h + gap_y / 2)
            y1 = int((i + 1) * row_h - gap_y / 2)

            self.c.coords(row["panel"], pad, y0 + pad // 2, W - pad, y1 - pad // 2)

            title_h = int((y1 - y0) * 0.18)
            self.c.coords(row["title_bar"], pad, y1 - title_h - pad // 2, W - pad, y1 - pad // 2)
            self.c.coords(row["title_txt"], W // 2, y1 - title_h // 2 - pad // 2)
            self._fit_bg_to_text(row["title_bg"], row["title_txt"], self.TITLE_PAD_X, self.TITLE_PAD_Y)

            content_y0 = y0 + pad
            content_y1 = y1 - title_h - pad
            content_h = content_y1 - content_y0

            hdr_h = int(content_h * 0.22)
            hdr_y0 = content_y0
            hdr_y1 = hdr_y0 + hdr_h
            hdr_cy = (hdr_y0 + hdr_y1) // 2

            val_y = hdr_y1 + int((content_h - hdr_h) * 0.55)

            # place non-time digits
            self.c.coords(row["month"], bounds[0][2], val_y)
            self.c.coords(row["day"], bounds[1][2], val_y)
            self.c.coords(row["year"], bounds[2][2], val_y)

            # --- Place HOUR / COLON / MIN with even spacing around the colon ---
            colon_x = bounds[3][1]  # boundary between hour & min columns
            dot_r = max(4, int(content_h * 0.035))
            dot_gap = int(dot_r * 2.1)
            SIDE_GAP = int(self.TIME_SIDE_GAP)

            self.c.update_idletasks()
            hour_w, _ = self._bbox_wh(row["hour"])
            min_w, _ = self._bbox_wh(row["min"])

            hour_cx = int((colon_x - SIDE_GAP) - (hour_w / 2))
            min_cx = int((colon_x + SIDE_GAP) + (min_w / 2))

            self.c.coords(row["hour"], hour_cx, val_y)
            self.c.coords(row["min"], min_cx, val_y)

            # colon dots centered exactly on colon_x
            self.c.coords(row["dot1"], colon_x - dot_r, val_y - dot_gap - dot_r,
                          colon_x + dot_r, val_y - dot_gap + dot_r)
            self.c.coords(row["dot2"], colon_x - dot_r, val_y + dot_gap - dot_r,
                          colon_x + dot_r, val_y + dot_gap + dot_r)

            # --- RECENTER HEADER LABELS to match the values ---
            header_centers = [bounds[0][2], bounds[1][2], bounds[2][2], hour_cx, min_cx]
            for (rect_id, text_id), cx in zip(row["hdr"], header_centers):
                self.c.coords(text_id, int(cx), int(hdr_cy))
                self._fit_bg_to_text(rect_id, text_id, self.HDR_PAD_X, self.HDR_PAD_Y)

            # ---- AM/PM: spacing + red backgrounds sized to word ----
            lamp_r = max(6, int(content_h * 0.055))
            stack_sep = int(lamp_r * self.AMPM_STACK_SEP_MULT)

            # Scaled centering: midpoint between YEAR and actual HOUR, plus a fraction of that gap
            gap_width = max(1, hour_cx - bounds[2][2])
            gap_cx = int((bounds[2][2] + hour_cx) / 2)
            lamp_cx = gap_cx + int(gap_width * float(self.AMPM_SHIFT_FRAC))

            lamp_y_am = val_y - int(stack_sep / 2)
            lamp_y_pm = val_y + int(stack_sep / 2)

            label_gap = max(4, int(lamp_r * 2.05))
            label_y_am = lamp_y_am - lamp_r - label_gap
            label_y_pm = lamp_y_pm - lamp_r - label_gap

            self.c.coords(row["txt_am"], lamp_cx, label_y_am)
            self.c.coords(row["txt_pm"], lamp_cx, label_y_pm)

            self._fit_bg_to_text(row["am_bg"], row["txt_am"], self.AMPM_PAD_X, self.AMPM_PAD_Y)
            self._fit_bg_to_text(row["pm_bg"], row["txt_pm"], self.AMPM_PAD_X, self.AMPM_PAD_Y)

            self.c.coords(row["lamp_am"], lamp_cx - lamp_r, lamp_y_am - lamp_r, lamp_cx + lamp_r, lamp_y_am + lamp_r)
            self.c.coords(row["lamp_pm"], lamp_cx - lamp_r, lamp_y_pm - lamp_r, lamp_cx + lamp_r, lamp_y_pm + lamp_r)

    # ----------------- ticking: time -----------------

    @staticmethod
    def _mon3(dt: datetime) -> str:
        return dt.strftime("%b").upper()

    @staticmethod
    def _ampm(dt: datetime) -> str:
        return dt.strftime("%p")

    @staticmethod
    def _hhmm_12h(dt: datetime) -> tuple[str, str]:
        return dt.strftime("%I"), dt.strftime("%M")

    def _set_row_time(self, row: dict, dt: datetime) -> None:
        self.c.itemconfigure(row["month"], text=self._mon3(dt))
        self.c.itemconfigure(row["day"], text=dt.strftime("%d"))
        self.c.itemconfigure(row["year"], text=dt.strftime("%Y"))

        hh, mm = self._hhmm_12h(dt)
        self.c.itemconfigure(row["hour"], text=hh)
        self.c.itemconfigure(row["min"], text=mm)

        am_on = (self._ampm(dt) == "AM")
        pm_on = not am_on
        self.c.itemconfigure(row["lamp_am"], fill=row["seg_color"] if am_on else self.COL_DARK)
        self.c.itemconfigure(row["lamp_pm"], fill=row["seg_color"] if pm_on else self.COL_DARK)

    def _update_time(self) -> None:
        now = datetime.now()
        rows = self._items["rows"]

        self._set_row_time(rows[0], self.destination_dt)
        self._set_row_time(rows[1], now)
        self._set_row_time(rows[2], now - timedelta(minutes=1))

        self.after(200, self._update_time)

    # ----------------- ticking: colon blink (synced) -----------------

    def _set_colon(self, row: dict, on: bool) -> None:
        col = row["seg_color"] if on else self.COL_DARK
        self.c.itemconfigure(row["dot1"], fill=col, outline=col)
        self.c.itemconfigure(row["dot2"], fill=col, outline=col)

    def _delay_to_next_half_second_ms(self) -> int:
        t = time.time()
        next_half = ((int(t * 2) + 1) / 2.0)
        return max(1, int((next_half - t) * 1000))

    def _blink_colons(self) -> None:
        t = time.time()
        phase = t - int(t)        # 0.0..1.0
        colon_on = (phase < 0.5)  # ON first half, OFF second half

        for r in self._items["rows"]:
            self._set_colon(r, colon_on)

        self.after(self._delay_to_next_half_second_ms(), self._blink_colons)


if __name__ == "__main__":
    TimeCircuitsClock().mainloop()

