#!/usr/bin/env python3
import re
import subprocess
from datetime import datetime
import tkinter as tk
import tkinter.font as tkfont


def get_primary_monitor_geometry() -> tuple[int, int, int, int]:
    try:
        out = subprocess.check_output(["xrandr", "--listmonitors"], text=True, stderr=subprocess.STDOUT)
    except Exception:
        return (0, 0, 0, 0)

    for line in out.splitlines():
        if "+*" in line:
            m = re.search(r"(\d+)/\d+x(\d+)/\d+\+(\d+)\+(\d+)", line)
            if m:
                return (int(m.group(1)), int(m.group(2)), int(m.group(3)), int(m.group(4)))

    for line in out.splitlines():
        m = re.search(r"(\d+)/\d+x(\d+)/\d+\+(\d+)\+(\d+)", line)
        if m:
            return (int(m.group(1)), int(m.group(2)), int(m.group(3)), int(m.group(4)))

    return (0, 0, 0, 0)


def pick_dseg7_family(root: tk.Tk) -> str:
    available = set(tkfont.families(root))
    for name in ("DSEG7 Classic Bold", "DSEG7 Classic", "DSEG7 Modern"):
        if name in available:
            return name
    return "DejaVu Sans Mono"


def pick_dseg14_family(root: tk.Tk) -> str:
    """
    14-segment font renders letters properly (e.g., 'SAT', 'JAN').
    Falls back to a normal sans font if DSEG14 isn't installed.
    """
    available = set(tkfont.families(root))
    for name in ("DSEG14 Classic Bold", "DSEG14 Classic", "DSEG14 Modern"):
        if name in available:
            return name
    return "DejaVu Sans"


class ClockDigitalNoGlow(tk.Tk):
    def __init__(self) -> None:
        super().__init__()

        # Hide while initial layout/fit happens (prevents visible "zoom")
        self.withdraw()

        # ---- TUNING KNOBS ----
        CLOCK_SCALE = 1.0
        AMPM_SCALE = 0.090          # AM/PM size
        DATE_GAP_MIN_PX = 40
        DATE_GAP_FRAC_W = 0.03

        # Quit gestures

        # Cursor hide/show (requested)
        CURSOR_HIDE_SECONDS = 7  # set 5..10 as you prefer

        self.configure(bg="black")

        # IMPORTANT (Linux PC behaviour):
        # Use Tk fullscreen instead of overrideredirect. This preserves normal focus/keyboard handling
        # under typical Linux desktop window managers, making Escape reliable.
        self.overrideredirect(False)
        self.attributes("-fullscreen", True)
        self.attributes("-topmost", True)

        w, h, x, y = get_primary_monitor_geometry()
        if w > 0 and h > 0:
            self.geometry(f"{w}x{h}+{x}+{y}")
        else:
            self.update_idletasks()
            w, h = self.winfo_screenwidth(), self.winfo_screenheight()
            self.geometry(f"{w}x{h}+0+0")

        self.update_idletasks()
        self.W, self.H = w, h

        self.c = tk.Canvas(self, bg="black", highlightthickness=0, bd=0)
        self.c.pack(fill="both", expand=True)

        self.RED = "#FF0000"
        self.RED_DIM = "#9A0000"

        dseg7 = pick_dseg7_family(self)
        dseg14 = pick_dseg14_family(self)

        # Fonts (digits)
        self.big_font = tkfont.Font(family=dseg7, size=int(h * 0.38 * CLOCK_SCALE), weight="bold")
        self.sec_font = tkfont.Font(family=dseg7, size=int(h * 0.25), weight="bold")
        self.small_date_font = tkfont.Font(family=dseg7, size=int(h * 0.14), weight="bold")
        self.small_year_font = tkfont.Font(family=dseg7, size=int(h * 0.14), weight="bold")

        # Fonts (letters): use 14-seg for DAY and MONTH values
        self.day_font = tkfont.Font(family=dseg14, size=int(h * 0.11), weight="bold")
        self.month_text_font = tkfont.Font(family=dseg14, size=int(h * 0.11), weight="bold")

        # Labels and AM/PM
        self.label_font = tkfont.Font(family="DejaVu Sans", size=int(h * 0.040), weight="bold")
        self.ampm_font = tkfont.Font(family="DejaVu Sans", size=max(10, int(h * AMPM_SCALE)), weight="bold")

        # Layout knobs
        self._date_gap_min_px = int(DATE_GAP_MIN_PX)
        self._date_gap_frac_w = float(DATE_GAP_FRAC_W)

        # Main time
        self.hh = self.c.create_text(0, 0, text="7", fill=self.RED, font=self.big_font, anchor="nw")
        self.colon = self.c.create_text(0, 0, text=":", fill=self.RED, font=self.big_font, anchor="nw")
        self.mm = self.c.create_text(0, 0, text="00", fill=self.RED, font=self.big_font, anchor="nw")

        self.ampm = self.c.create_text(0, 0, text="AM", fill=self.RED_DIM, font=self.ampm_font, anchor="w")

        # Seconds (top-right)
        self.ss = self.c.create_text(0, 0, text="00", fill=self.RED, font=self.sec_font, anchor="ne")

        # Bottom-right cluster: DAY / DATE / MONTH / YEAR
        self.dow2_lbl = self.c.create_text(0, 0, text="DAY", fill=self.RED_DIM, font=self.label_font, anchor="center")
        self.date_lbl = self.c.create_text(0, 0, text="DATE", fill=self.RED_DIM, font=self.label_font, anchor="center")
        self.month_lbl = self.c.create_text(0, 0, text="MONTH", fill=self.RED_DIM, font=self.label_font, anchor="center")
        self.year_lbl = self.c.create_text(0, 0, text="YEAR", fill=self.RED_DIM, font=self.label_font, anchor="center")

        self.dow2_val = self.c.create_text(0, 0, text="MON", fill=self.RED, font=self.day_font, anchor="center")
        self.date_val = self.c.create_text(0, 0, text="1", fill=self.RED, font=self.small_date_font, anchor="center")
        self.month_val = self.c.create_text(0, 0, text="JAN", fill=self.RED, font=self.month_text_font, anchor="center")
        self.year_val = self.c.create_text(0, 0, text="2026", fill=self.RED, font=self.small_year_font, anchor="center")

        # Cursor hide/show management
        self._cursor_hide_ms = int(CURSOR_HIDE_SECONDS * 1000)
        self._cursor_after_id = None
        self._cursor_hidden = False
        self.bind_all("<Motion>", self._on_mouse_motion)

        # Initial hidden render + fit
        self._tick(initial=True)
        self._fit_to_screen()

        # Show window, then arm input handlers
        self.update_idletasks()
        self.deiconify()
        self.after(50, self._arm_quit_controls)

        # Hide cursor after idle
        self._schedule_cursor_hide()

        # Start normal ticking
        self._tick(initial=False)

    # ----- Cursor hide/show -----

    def _set_cursor_hidden(self, hidden: bool) -> None:
        if hidden and not self._cursor_hidden:
            self.configure(cursor="none")
            self.c.configure(cursor="none")
            self._cursor_hidden = True
        elif not hidden and self._cursor_hidden:
            self.configure(cursor="")
            self.c.configure(cursor="")
            self._cursor_hidden = False

    def _schedule_cursor_hide(self) -> None:
        if self._cursor_after_id is not None:
            try:
                self.after_cancel(self._cursor_after_id)
            except Exception:
                pass
        self._cursor_after_id = self.after(self._cursor_hide_ms, lambda: self._set_cursor_hidden(True))

    def _on_mouse_motion(self, _evt=None) -> None:
        self._set_cursor_hidden(False)
        self._schedule_cursor_hide()

    # ----- Quit controls -----

    def _arm_quit_controls(self) -> None:
        # Make sure we can receive keypresses under a normal Linux WM
        try:
            self.lift()
            self.focus_force()
            self.c.focus_set()
        except tk.TclError:
            pass

        def _quit(_e=None):
            self.destroy()

        # Escape (bind on root + bind_all for reliability)
        self.bind("<Escape>", _quit)
        self.bind("<KeyPress-Escape>", _quit)
        self.bind_all("<Escape>", _quit)
        self.bind_all("<KeyPress-Escape>", _quit)

        # Optional: keep q
        self.bind("q", _quit)
        self.bind_all("q", _quit)

        # Right click exit
        self.bind("<Button-3>", _quit)
        self.c.bind("<Button-3>", _quit)

    # ----- Layout helpers -----

    def _bbox_wh_item(self, item_id: int) -> tuple[int, int]:
        bb = self.c.bbox(item_id)
        if not bb:
            return (0, 0)
        x0, y0, x1, y1 = bb
        return (x1 - x0, y1 - y0)

    def _shrink_font(self, f: tkfont.Font, factor: float, min_size: int = 6) -> None:
        size = int(max(min_size, round(f.cget("size") * factor)))
        f.configure(size=size)

    def _fit_to_screen(self) -> None:
        pad = int(min(self.W, self.H) * 0.02)
        max_tries = 25

        for _ in range(max_tries):
            self.c.update_idletasks()
            bb = self.c.bbox("all")
            if not bb:
                return
            x0, y0, x1, y1 = bb

            if x0 >= pad and y0 >= pad and x1 <= (self.W - pad) and y1 <= (self.H - pad):
                return

            # Shrink only the small clusters
            self._shrink_font(self.sec_font, 0.96)
            self._shrink_font(self.small_date_font, 0.96)
            self._shrink_font(self.small_year_font, 0.96)
            self._shrink_font(self.day_font, 0.96)
            self._shrink_font(self.month_text_font, 0.96)
            self._shrink_font(self.label_font, 0.96)
            self._shrink_font(self.ampm_font, 0.96)

            self._layout()

    def _layout(self) -> None:
        W, H = self.W, self.H
        margin = int(W * 0.04)

        # Centre HH:MM group
        hh_w, hh_h = self._bbox_wh_item(self.hh)
        col_w, col_h = self._bbox_wh_item(self.colon)
        mm_w, mm_h = self._bbox_wh_item(self.mm)

        group_w = hh_w + col_w + mm_w
        group_h = max(hh_h, col_h, mm_h)

        gx0 = int((W - group_w) / 2)
        gy0 = int((H - group_h) / 2)

        self.c.coords(self.hh, gx0, gy0)
        self.c.coords(self.colon, gx0 + hh_w, gy0)
        self.c.coords(self.mm, gx0 + hh_w + col_w, gy0)

        # AM/PM left of HH:MM
        am_w, _ = self._bbox_wh_item(self.ampm)
        self.c.coords(self.ampm, max(margin, gx0 - am_w - int(W * 0.02)), gy0 + int(group_h * 0.33))

        # Seconds top-right
        right_x = W - margin
        self.c.coords(self.ss, right_x, margin)

        # Bottom-right cluster
        dow_w, _ = self._bbox_wh_item(self.dow2_val)
        date_w, _ = self._bbox_wh_item(self.date_val)
        month_w, _ = self._bbox_wh_item(self.month_val)
        year_w, _ = self._bbox_wh_item(self.year_val)

        gap = max(int(W * self._date_gap_frac_w), self._date_gap_min_px)

        col4_x = right_x - int(W * 0.03)  # YEAR (rightmost)
        col3_x = col4_x - (year_w // 2) - gap - (month_w // 2)  # MONTH
        col2_x = col3_x - (month_w // 2) - gap - (date_w // 2)  # DATE
        col1_x = col2_x - (date_w // 2) - gap - (dow_w // 2)    # DAY

        bottom_val_y = H - margin - int(H * 0.01)
        bottom_lbl_y = bottom_val_y - int(H * 0.09)

        self.c.coords(self.dow2_lbl, col1_x, bottom_lbl_y)
        self.c.coords(self.date_lbl, col2_x, bottom_lbl_y)
        self.c.coords(self.month_lbl, col3_x, bottom_lbl_y)
        self.c.coords(self.year_lbl, col4_x, bottom_lbl_y)

        self.c.coords(self.dow2_val, col1_x, bottom_val_y)
        self.c.coords(self.date_val, col2_x, bottom_val_y)
        self.c.coords(self.month_val, col3_x, bottom_val_y)
        self.c.coords(self.year_val, col4_x, bottom_val_y)

    # ----- Tick -----

    def _tick(self, initial: bool = False) -> None:
        now = datetime.now()

        # No leading zero for 1..9; keep 10/11/12
        hh_text = str(int(now.strftime("%I")))
        mm = now.strftime("%M")
        ss = now.strftime("%S")
        ampm = now.strftime("%p")

        dow = now.strftime("%a").upper()
        day_num = str(int(now.strftime("%d")))
        mon3 = now.strftime("%b").upper()
        year4 = now.strftime("%Y")

        self.c.itemconfigure(self.hh, text=hh_text)
        self.c.itemconfigure(self.mm, text=mm)
        self.c.itemconfigure(self.ss, text=ss)
        self.c.itemconfigure(self.ampm, text=ampm)

        self.c.itemconfigure(self.dow2_val, text=dow)
        self.c.itemconfigure(self.date_val, text=day_num)
        self.c.itemconfigure(self.month_val, text=mon3)
        self.c.itemconfigure(self.year_val, text=year4)

        self._layout()

        if not initial:
            delay = max(1, 1000 - int(now.microsecond / 1000))
            self.after(delay, self._tick)


if __name__ == "__main__":
    ClockDigitalNoGlow().mainloop()

