#!/usr/bin/env python3
import tkinter as tk
import tkinter.font as tkfont
from datetime import datetime


class FullscreenClock(tk.Tk):
    def __init__(self) -> None:
        super().__init__()

        self.configure(bg="black")
        self.attributes("-fullscreen", True)
        self.attributes("-topmost", True)

        # Exit controls
        self.after(50, self._claim_input)
        self.bind("<Escape>", lambda e: self.destroy())
        self.bind_all("<Escape>", lambda e: self.destroy())
        self.bind("<Button-3>", lambda e: self.destroy())
        self.bind_all("<Button-3>", lambda e: self.destroy())
        self.bind("<Button-2>", lambda e: self.destroy())
        self.bind_all("<Button-2>", lambda e: self.destroy())

        self.update_idletasks()
        self.w = self.winfo_screenwidth()
        self.h = self.winfo_screenheight()

        fg = "#A0A0A0"
        bg = "black"

        # ---------- Size model ----------
        # Target usable area (leave a small margin so nothing clips)
        margin_x = int(self.w * 0.04)
        usable_w = max(200, self.w - (2 * margin_x))

        # Ratios (relative to main time font)
        SEC_RATIO = 0.33
        AM_RATIO  = 0.18
        S_RATIO   = 0.16

        # Padding between HH:MM and seconds group (relative to time font)
        PAD_RATIO = 0.06

        # Fit the main time font so the *whole* row fits
        # Worst-case for width is "12:34" and seconds "59"
        sample_hhmm = "12:34"
        sample_ss = "59"

        # Start from a sensible height-based guess, then fit to width
        start_size = int(self.h * 0.35)  # not huge
        min_size = 18

        time_size = self._fit_time_size(
            family="DejaVu Sans",
            weight="bold",
            hhmm_sample=sample_hhmm,
            ss_sample=sample_ss,
            usable_w=usable_w,
            start_size=start_size,
            min_size=min_size,
            sec_ratio=SEC_RATIO,
            s_ratio=S_RATIO,
            pad_ratio=PAD_RATIO,
        )

        sec_size = max(10, int(time_size * SEC_RATIO))
        am_size  = max(10, int(time_size * AM_RATIO))
        s_size   = max(10, int(time_size * S_RATIO))
        date_size = max(12, int(time_size * 0.18))

        self.font_time = ("DejaVu Sans", time_size, "bold")
        self.font_sec  = ("DejaVu Sans", sec_size, "bold")
        self.font_am   = ("DejaVu Sans", am_size, "bold")
        self.font_s    = ("DejaVu Sans", s_size, "bold")
        self.font_date = ("DejaVu Sans", date_size, "normal")

        # ---------- Layout ----------
        root_frame = tk.Frame(self, bg="black")
        root_frame.pack(expand=True, fill="both")

        block = tk.Frame(root_frame, bg="black")
        block.place(relx=0.5, rely=0.48, anchor="center")

        # AM/PM in the top-left of the time area
        self.lbl_am = tk.Label(block, text="", font=self.font_am, fg=fg, bg=bg)
        self.lbl_am.place(x=0, y=0, anchor="nw")

        # Main row
        row = tk.Frame(block, bg="black")
        # Push down so AM/PM sits above
        row.pack(pady=(int(time_size * 0.22), 0))

        self.lbl_hhmm = tk.Label(row, text="--:--", font=self.font_time, fg=fg, bg=bg)
        self.lbl_hhmm.pack(side="left", anchor="s")

        sec_frame = tk.Frame(row, bg="black")
        sec_frame.pack(side="left", padx=(int(time_size * PAD_RATIO), 0), anchor="n")

        self.lbl_ss = tk.Label(sec_frame, text="--", font=self.font_sec, fg=fg, bg=bg)
        self.lbl_ss.pack(side="top", anchor="e", pady=(int(time_size * 0.10), 0))

        self.lbl_date = tk.Label(block, text="", font=self.font_date, fg="#7A0000", bg="black")
        self.lbl_date.pack(pady=(int(time_size * 0.16), 0))

        self._tick()

    def _claim_input(self) -> None:
        try:
            self.focus_force()
            self.focus_set()
            self.lift()
            self.grab_set()
        except tk.TclError:
            pass

    def _fit_time_size(
        self,
        family: str,
        weight: str,
        hhmm_sample: str,
        ss_sample: str,
        usable_w: int,
        start_size: int,
        min_size: int,
        sec_ratio: float,
        s_ratio: float,
        pad_ratio: float,
    ) -> int:
        """
        Finds largest time font size such that:
        width(hhmm) + pad + max(width("S"), width(ss)) <= usable_w
        """
        size = max(min_size, start_size)

        while size >= min_size:
            f_time = tkfont.Font(root=self, family=family, size=size, weight=weight)
            f_sec  = tkfont.Font(root=self, family=family, size=max(10, int(size * sec_ratio)), weight=weight)
            f_s    = tkfont.Font(root=self, family=family, size=max(10, int(size * s_ratio)), weight=weight)

            hhmm_w = f_time.measure(hhmm_sample)
            pad_px = int(size * pad_ratio)
            sec_w = max(f_sec.measure(ss_sample), f_s.measure("S"))

            total = hhmm_w + pad_px + sec_w

            if total <= usable_w:
                return size

            size -= 2

        return min_size

    def _tick(self) -> None:
        now = datetime.now()
        hh = now.strftime("%I").lstrip("0") or "12"
        mm = now.strftime("%M")
        ss = now.strftime("%S")
        am = now.strftime("%p")

        self.lbl_hhmm.config(text=f"{hh}:{mm}")
        self.lbl_ss.config(text=ss)
        self.lbl_am.config(text=am)

        self.lbl_date.config(text=now.strftime("%A %d %B %Y"))

        delay = max(1, 1000 - int(now.microsecond / 1000))
        self.after(delay, self._tick)


if __name__ == "__main__":
    FullscreenClock().mainloop()

